import asyncio
import ipaddress
from datetime import datetime
from fastapi import FastAPI, WebSocket, status
from fastapi.responses import HTMLResponse
from fastapi.middleware.cors import CORSMiddleware

app = FastAPI()

# إعدادات متقدمة للاتصالات
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# إعدادات الأمان
ALLOWED_IPS = {
    "46.184.88.178/32",
    "154.239.184.76/32",
    "192.168.1.0/24"
}
PTS_HANDSHAKE_FORMAT = "PTS2/1.0 CONNECT|DEVICE_ID={}|VERSION={}"

# حالة الأجهزة
connected_devices = {}
lock = asyncio.Lock()

@app.websocket("/ws")
async def pts_websocket(websocket: WebSocket):
    client_ip = websocket.client.host
    
    # التحقق من IP مع طباعة السجل
    if not is_ip_allowed(client_ip):
        print(f"IP Blocked: {client_ip}")
        await websocket.close(code=status.WS_1008_POLICY_VIOLATION)
        return
    
    # قبول الاتصال مع البروتوكول المطلوب
    await websocket.accept(subprotocol="pts-v1")
    
    try:
        # استقبال بيانات المصافحة مع مهلة زمنية
        handshake = await asyncio.wait_for(websocket.receive_text(), timeout=5.0)
        
        if not handshake.startswith("PTS2/1.0 CONNECT|"):
            await websocket.close(code=status.WS_1003_UNSUPPORTED_DATA)
            return
            
        # استخراج بيانات الجهاز
        device_id = handshake.split("|")[1].split("=")[1]
        
        # التسجيل في السجلات
        print(f"New connection: {device_id} from {client_ip}")
        
        # إرسال تأكيد الاتصال
        await websocket.send_text("PTS2/1.0 ACCEPTED")
        
        # حلقة الاستماع
        while True:
            data = await websocket.receive_text()
            if data == "PING":
                await websocket.send_text("PONG")
                
    except asyncio.TimeoutError:
        print("Handshake timeout")
        await websocket.close(code=status.WS_1008_POLICY_VIOLATION)
    except WebSocketDisconnect:
        print(f"Disconnected: {device_id}")
    except Exception as e:
        print(f"Error: {str(e)}")
    finally:
        await websocket.close()

def is_ip_allowed(ip: str) -> bool:
    try:
        ip_obj = ipaddress.ip_address(ip)
        return any(
            ip_obj in ipaddress.ip_network(net.strip(), strict=False)
            for net in ALLOWED_IPS
        )
    except ValueError:
        return False

def parse_handshake(data: str) -> tuple:
    """تحليل بيانات المصافحة مع معالجة الأخطاء"""
    try:
        if not data.startswith("PTS2/1.0 CONNECT|"):
            return None, None
            
        parts = data.split("|")
        device_id = parts[1].split("=")[1]
        version = parts[2].split("=")[1]
        return device_id, version
    except (IndexError, ValueError):
        return None, None

@app.get("/", response_class=HTMLResponse)
async def monitoring_interface():
    return """
    <html>
        <style>
            .device {padding: 10px; margin: 5px; border: 1px solid #ccc;}
            .connected {background: #dfd;}
            .disconnected {background: #fdd;}
        </style>
        <body>
            <h1>مراقبة أجهزة PTS-2</h1>
            <div id="devices"></div>
            <script>
                async function update() {
                    const res = await fetch('/devices');
                    const devices = await res.json();
                    let html = '';
                    devices.forEach(d => {
                        html += `<div class="device ${d.status}">
                            <h3>${d.id}</h3>
                            <p>الإصدار: ${d.version}</p>
                            <p>الحالة: ${d.status}</p>
                            <p>آخر نشاط: ${new Date(d.last_seen).toLocaleString()}</p>
                        </div>`;
                    });
                    document.getElementById('devices').innerHTML = html;
                }
                setInterval(update, 2000);
                update();
            </script>
        </body>
    </html>
    """

@app.get("/devices")
async def get_devices():
    async with lock:
        now = datetime.utcnow()
        return [
            {
                "id": did,
                "version": data["version"],
                "status": "connected" if (now - data["last_heartbeat"]).seconds < 60 else "disconnected",
                "last_seen": data["last_heartbeat"],
                "ip": data["ip"]
            }
            for did, data in connected_devices.items()
        ]

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(
        app,
        host="0.0.0.0",
        port=5000,
        ws_ping_interval=30,
        ws_ping_timeout=60
    )