<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Ratchet\MessageComponentInterface;
use Ratchet\ConnectionInterface;

class WebSocketController  implements MessageComponentInterface
{
    protected $clients;
    protected $connectedDevices = [];
    protected $devicesStatus = [];
    protected $lock;
    protected $lastStatusRequest = [];

    public function __construct()
    {
        $this->clients = new \SplObjectStorage;
        $this->lock = new \React\Promise\Deferred();
    }

    public function index()
    {
        return response()->file(public_path('index.html'));
    }

    public function onOpen(ConnectionInterface $conn)
    {
        $this->clients->attach($conn);
        $this->connectedDevices[$conn->resourceId] = $conn;
        
        // Extract IP address from the connection
        $ip = $conn->remoteAddress ?? 'غير معروف';
        Log::info("New connection open ! ({$conn->resourceId}) from IP: {$ip}");

        // Send initial connection confirmation
        $conn->send(json_encode([
            'type' => 'connection',
            'status' => 'connected',
            'client_id' => $conn->resourceId,
            'ip_address' => $ip
        ]));
    }

    public function onMessage(ConnectionInterface $from, $msg)
    {
        // Extract IP address from the connection
        $ip = $from->remoteAddress ?? 'غير معروف';
        echo "Received message from {$from->resourceId} (IP: {$ip}): {$msg}\n";
        Log::info("Received message from ({$from->resourceId}, IP: {$ip}): {$msg}");

        try {
            $data = json_decode($msg, true);
            
            // Handle PTS Protocol message
            if (isset($data['Protocol']) && $data['Protocol'] === 'jsonPTS') {
                $deviceId = $data['PtsId'] ?? $from->resourceId;
                $data['ip_address'] = $ip; // Add IP address to the PTS data
                
                // Check for error message from PTS
                if (isset($data['Error']) && $data['Error'] === true) {
                    $errorMessage = $data['Message'] ?? 'Unknown error';
                    Log::warning("PTS Error from device {$deviceId}: {$errorMessage}");
                    
                    // Create a response in exact jsonPTS format the device expects
                    $ptsResponse = [
                        'Protocol' => 'jsonPTS',
                        'PtsId' => $deviceId,
                        'Result' => 'OK'  // This is a critical field expected by the PTS device
                    ];
                    
                    $from->send(json_encode($ptsResponse));
                    Log::info("Sent PTS error acknowledgment in jsonPTS format: " . json_encode($ptsResponse));
                    
                    // Still update the device status even if there's an error
                    if (isset($this->devicesStatus[$deviceId])) {
                        $this->devicesStatus[$deviceId]['last_seen'] = now();
                        $this->devicesStatus[$deviceId]['status'] = 'متصل';
                        $this->devicesStatus[$deviceId]['ip_address'] = $ip;
                    }
                } else {
                    // Initialize device status if it doesn't exist
                    if (!isset($this->devicesStatus[$deviceId])) {
                        $this->devicesStatus[$deviceId] = [
                            'device_id' => $deviceId,
                            'serial_number' => $deviceId,
                            'mac_address' => 'N/A',
                            'ip_address' => $ip,
                            'status' => 'متصل',
                            'last_seen' => now(),
                            'pts_info' => [
                                'device_type' => 'PTS Device',
                                'firmware_version' => 'Unknown',
                                'operation_status' => 'متصل',
                                'battery' => 0,
                                'temperature' => 0,
                                'fuel_grades' => []
                            ],
                            'transactions' => []
                        ];
                    }
                    
                    // Always update the last_seen timestamp
                    $this->devicesStatus[$deviceId]['last_seen'] = now();
                    $this->devicesStatus[$deviceId]['ip_address'] = $ip;
                    $this->devicesStatus[$deviceId]['status'] = 'متصل';
                    
                    // Process packets and build response
                    $processedPackets = [];
                    $hasProcessedPackets = false;
                    $receivedPumpTransaction = false;
                    
                    if (isset($data['Packets']) && is_array($data['Packets'])) {
                        foreach ($data['Packets'] as $packet) {
                            // Check if packet has Type field
                            if (!isset($packet['Type']) || !isset($packet['Data'])) {
                                Log::warning("Invalid packet received from device {$deviceId}: Missing Type or Data");
                                continue;
                            }
                            
                            $hasProcessedPackets = true;
                            $packetId = $packet['Id'] ?? null;
                            $packetType = $packet['Type'];
                            
                            // Log each packet for debugging
                            Log::info("Processing packet from PTS {$deviceId}: Type={$packetType}, ID={$packetId}");
                            
                            // Add this packet to processed list
                            $processedPackets[] = [
                                'Id' => $packetId,
                                'Type' => $packetType,
                                'Result' => 'OK'
                            ];
                            
                            // Process each packet type independently
                            if ($packet['Type'] === 'UploadStatus') {
                                $statusData = $packet['Data'];
                                
                                $this->devicesStatus[$deviceId]['pts_info'] = [
                                    'device_type' => 'PTS Device',
                                    'firmware_version' => $statusData['FirmwareDateTime'] ?? 'Unknown',
                                    'operation_status' => 'متصل',
                                    'battery' => isset($statusData['BatteryVoltage']) ? $statusData['BatteryVoltage'] / 1000 : 0,
                                    'temperature' => $statusData['CpuTemperature'] ?? 0,
                                    'fuel_grades' => $statusData['FuelGrades'] ?? []
                                ];
                                
                                // Log status update
                                Log::info("Status update received from PTS device {$deviceId}");
                            }
                            
                            if ($packet['Type'] === 'Transaction' || $packet['Type'] === 'UploadPumpTransaction') {
                                $receivedPumpTransaction = true;
                                $transactionData = $packet['Data'];
                                
                                // Store the transaction with all required fields
                                $transaction = [
                                    'transaction_id' => $transactionData['Transaction'] ?? $transactionData['Id'] ?? uniqid(),
                                    'timestamp' => $transactionData['DateTime'] ?? $transactionData['Timestamp'] ?? now(),
                                    'data' => $transactionData,
                                    'received_at' => now(),
                                    'type' => $packet['Type']
                                ];
                                
                                // Initialize transactions array if needed
                                if (!isset($this->devicesStatus[$deviceId]['transactions'])) {
                                    $this->devicesStatus[$deviceId]['transactions'] = [];
                                }
                                
                                // Check for duplicate transactions
                                $isDuplicate = false;
                                foreach ($this->devicesStatus[$deviceId]['transactions'] as $existingTransaction) {
                                    if ($existingTransaction['transaction_id'] == $transaction['transaction_id']) {
                                        $isDuplicate = true;
                                        break;
                                    }
                                }
                                
                                if (!$isDuplicate) {
                                    array_unshift($this->devicesStatus[$deviceId]['transactions'], $transaction);
                                    
                                    // Keep only the last 50 transactions
                                    if (count($this->devicesStatus[$deviceId]['transactions']) > 50) {
                                        array_pop($this->devicesStatus[$deviceId]['transactions']);
                                    }
                                    
                                    // Log transaction received
                                    Log::info("Transaction data received from PTS device {$deviceId}: Type " . $packet['Type'] . ", ID: " . $transaction['transaction_id']);
                                } else {
                                    Log::info("Duplicate transaction ignored from PTS device {$deviceId}: ID " . $transaction['transaction_id']);
                                }
                            }
                        }
                    }
                    
                    // Create base PTS protocol response
                    $ptsResponse = [
                        'Protocol' => 'jsonPTS',
                        'PtsId' => $deviceId,
                        'Result' => 'OK'
                    ];
                    
                    // Add processed packets info if there were any
                    if ($hasProcessedPackets) {
                        $ptsResponse['Packets'] = $processedPackets;
                    }
                    
                    // IMPORTANT: Always request status if we received a pump transaction
                    // This ensures we get status after each transaction
                    if ($receivedPumpTransaction) {
                        $ptsResponse['Requests'] = [
                            [
                                'Type' => 'GetStatus',
                                'Id' => rand(1000, 9999)
                            ]
                        ];
                        Log::info("Requesting status update because we received a pump transaction from device {$deviceId}");
                    }
                    
                    // Log and send response
                    $responseJson = json_encode($ptsResponse);
                    $from->send($responseJson);
                    Log::info("Sent PTS response: {$responseJson}");
                    
                    // Broadcast device status update to web clients
                    $statusUpdate = [
                        'type' => 'device_status_update',
                        'device' => $this->devicesStatus[$deviceId]
                    ];
                    
                    foreach ($this->clients as $client) {
                        if ($from !== $client) {
                            $client->send(json_encode($statusUpdate));
                        }
                    }
                }
                
                // Completed PTS processing
                return;
            }
            
            // Handle action requests from web clients
            if (isset($data['action'])) {
                switch ($data['action']) {
                    case 'getDevices':
                        // Send the device list to the requester
                        $response = [
                            'type' => 'devices',
                            'devices' => array_values($this->devicesStatus)
                        ];
                        $from->send(json_encode($response));
                        return;
                }
            }
            
            // Handle device info registration
            $deviceId = $data['device_id'] ?? $from->resourceId;

            if (isset($data['device_info'])) {
                $this->devicesStatus[$deviceId] = [
                    'device_id' => $deviceId,
                    'serial_number' => $data['device_info']['serial_number'] ?? 'غير متوفر',
                    'mac_address' => $data['device_info']['mac_address'] ?? 'غير متوفر',
                    'ip_address' => $ip,  // Store IP address
                    'status' => 'متصل',
                    'last_seen' => now(),
                    'pts_info' => [
                        'device_type' => 'بانتظار المعلومات',
                        'firmware_version' => 'بانتظار المعلومات',
                        'operation_status' => 'متصل'
                    ],
                    'transactions' => []
                ];
                
                // Send confirmation back to sender with IP address
                $from->send(json_encode([
                    'type' => 'device_registered',
                    'device_id' => $deviceId,
                    'ip_address' => $ip,
                    'status' => 'success'
                ]));
            }

            // Broadcast to other clients (for non-PTS protocol messages)
            foreach ($this->clients as $client) {
                if ($from !== $client) {
                    // Add IP address to broadcast messages
                    if (!isset($data['ip_address'])) {
                        $data['ip_address'] = $ip;
                    }
                    $client->send(json_encode($data));
                }
            }
        } catch (\Exception $e) {
            echo "Error processing message: " . $e->getMessage() . "\n";
            Log::error("WebSocket error: " . $e->getMessage() . " for message: " . substr($msg, 0, 200));
            
            // Try to send an acknowledgment back even if there was an error
            try {
                // Check if this was a PTS request and respond with proper format if it was
                if (isset($data) && isset($data['Protocol']) && $data['Protocol'] === 'jsonPTS') {
                    $deviceId = $data['PtsId'] ?? ($from->resourceId ?? 'unknown');
                    $from->send(json_encode([
                        'Protocol' => 'jsonPTS',
                        'PtsId' => $deviceId,
                        'Result' => 'Error',
                        'Message' => 'Server error processing request'
                    ]));
                } else {
                    $from->send(json_encode([
                        'type' => 'error',
                        'message' => 'Error processing your message',
                        'status' => 'error'
                    ]));
                }
            } catch (\Exception $sendError) {
                Log::error("Failed to send error response: " . $sendError->getMessage());
            }
        }
    }

    public function onClose(ConnectionInterface $conn)
    {
        $this->clients->detach($conn);
        Log::info("Connection {$conn->resourceId} has disconnected");
    }

    public function onError(ConnectionInterface $conn, \Exception $e)
    {
        Log::error("An error has occurred: {$e->getMessage()}");
        $conn->close();
    }

    public function getDevices()
    {
        try {
            $devices = array_values($this->devicesStatus);
            return response()->json([
                'status' => 'success',
                'data' => $devices
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function disconnectDevice($deviceId)
    {
        if (isset($this->connectedDevices[$deviceId])) {
            $this->connectedDevices[$deviceId]->close();
            unset($this->connectedDevices[$deviceId]);
            $this->devicesStatus[$deviceId]['status'] = 'غير متصل';
            $this->devicesStatus[$deviceId]['last_seen'] = now();
            return response()->json(['status' => 'success', 'message' => "تم قطع الاتصال للجهاز $deviceId"]);
        }
        return response()->json(['status' => 'error', 'message' => 'الجهاز غير متصل']);
    }

    public function sendCommand($deviceId, Request $request)
    {
        $data = $request->json()->all();
        $command = $data['command'] ?? '';

        if (isset($this->connectedDevices[$deviceId])) {
            $this->connectedDevices[$deviceId]->send(json_encode($data));
            return response()->json(['status' => 'success', 'message' => "تم إرسال الأمر $command"]);
        }
        return response()->json(['status' => 'error', 'message' => 'الجهاز غير متصل']);
    }

    public function getReceipts($deviceId)
    {
        // Implement getReceipts logic
    }

    public function getTankInfo($deviceId)
    {
        // Implement getTankInfo logic
    }

    public function getPrices($deviceId)
    {
        // Implement getPrices logic
    }

    public function updatePrices($deviceId, Request $request)
    {
        // Implement updatePrices logic
    }

    public function getPtsStatus($deviceId)
    {
        // Implement getPtsStatus logic
    }

    public function manageShift($deviceId, Request $request)
    {
        // Implement manageShift logic
    }

    public function controlPump($deviceId, Request $request)
    {
        // Implement controlPump logic
    }

    public function getDailyReport($deviceId)
    {
        // Implement getDailyReport logic
    }
}
