<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Ratchet\MessageComponentInterface;
use Ratchet\ConnectionInterface;

class WebSocketController extends Controller implements MessageComponentInterface
{
    protected $clients;
    protected $connectedDevices = [];
    protected $devicesStatus = [];
    protected $lock;

    public function __construct()
    {
        $this->clients = new \SplObjectStorage;
        $this->lock = new \React\Promise\Deferred();
    }

    public function index()
    {
        return response()->file(public_path('index.html'));
    }

    public function onOpen(ConnectionInterface $conn)
    {
        $this->clients->attach($conn);
        $this->connectedDevices[$conn->resourceId] = $conn;
        Log::info("New connection! ({$conn->resourceId})");

        // Send initial connection confirmation
        $conn->send(json_encode([
            'type' => 'connection',
            'status' => 'connected',
            'client_id' => $conn->resourceId
        ]));
    }

    public function onMessage(ConnectionInterface $from, $msg)
    {
        echo "Received message from {$from->resourceId}: {$msg}\n";
        
        try {
            $data = json_decode($msg, true);
            
            if (isset($data['Protocol']) && $data['Protocol'] === 'jsonPTS') {
                $packet = $data['Packets'][0];
                
                // Handle price updates
                if ($packet['Type'] === 'SetFuelGradesConfiguration') {
                    Log::info('Updating fuel prices', $packet['Data']);
                    
                    // Broadcast to all clients
                    foreach ($this->clients as $client) {
                        $client->send($msg);
                    }
                }
                
                // Store device status
                if (isset($data['PtsId'])) {
                    $deviceId = $data['PtsId'];
                    $this->devicesStatus[$deviceId] = [
                        'last_seen' => now(),
                        'data' => $packet['Data']
                    ];
                }
            }
            
            $deviceId = $data['device_id'] ?? $from->resourceId;

            if (isset($data['device_info'])) {
                $this->devicesStatus[$deviceId] = [
                    'device_id' => $deviceId,
                    'serial_number' => $data['device_info']['serial_number'] ?? 'غير متوفر',
                    'mac_address' => $data['device_info']['mac_address'] ?? 'غير متوفر',
                    'status' => 'متصل',
                    'last_seen' => now(),
                    'pts_info' => [
                        'device_type' => 'بانتظار المعلومات',
                        'firmware_version' => 'بانتظار المعلومات',
                        'operation_status' => 'متصل'
                    ]
                ];
                
                // Send confirmation back to sender
                $from->send(json_encode([
                    'type' => 'device_registered',
                    'device_id' => $deviceId,
                    'status' => 'success'
                ]));
            }

            // Broadcast to other clients
            foreach ($this->clients as $client) {
                if ($from !== $client) {
                    $client->send($msg);
                }
            }
        } catch (\Exception $e) {
            echo "Error processing message: " . $e->getMessage() . "\n";
            Log::error("WebSocket error: " . $e->getMessage());
        }
    }

    public function onClose(ConnectionInterface $conn)
    {
        $this->clients->detach($conn);
        Log::info("Connection {$conn->resourceId} has disconnected");
    }

    public function onError(ConnectionInterface $conn, \Exception $e)
    {
        Log::error("An error has occurred: {$e->getMessage()}");
        $conn->close();
    }

    public function getDevices()
    {
        try {
            $devices = array_values($this->devicesStatus);
            return response()->json([
                'status' => 'success',
                'data' => $devices
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function disconnectDevice($deviceId)
    {
        if (isset($this->connectedDevices[$deviceId])) {
            $this->connectedDevices[$deviceId]->close();
            unset($this->connectedDevices[$deviceId]);
            $this->devicesStatus[$deviceId]['status'] = 'غير متصل';
            $this->devicesStatus[$deviceId]['last_seen'] = now();
            return response()->json(['status' => 'success', 'message' => "تم قطع الاتصال للجهاز $deviceId"]);
        }
        return response()->json(['status' => 'error', 'message' => 'الجهاز غير متصل']);
    }

    public function sendCommand($deviceId, Request $request)
    {
        $data = $request->json()->all();
        $command = $data['command'] ?? '';

        if (isset($this->connectedDevices[$deviceId])) {
            $this->connectedDevices[$deviceId]->send(json_encode($data));
            return response()->json(['status' => 'success', 'message' => "تم إرسال الأمر $command"]);
        }
        return response()->json(['status' => 'error', 'message' => 'الجهاز غير متصل']);
    }

    public function getReceipts($deviceId)
    {
        // Implement getReceipts logic
    }

    public function getTankInfo($deviceId)
    {
        // Implement getTankInfo logic
    }

    public function getPrices($deviceId)
    {
        // Implement getPrices logic
    }

    public function updatePrices($deviceId, Request $request)
    {
        // Implement updatePrices logic
    }

    public function getPtsStatus($deviceId)
    {
        // Implement getPtsStatus logic
    }

    public function manageShift($deviceId, Request $request)
    {
        // Implement manageShift logic
    }

    public function controlPump($deviceId, Request $request)
    {
        // Implement controlPump logic
    }

    public function getDailyReport($deviceId)
    {
        // Implement getDailyReport logic
    }
}